<?php
/**
 * Classe responsable de l'importation des commentaires dans GraphComment
 */

require_once __DIR__ . '/gc_comment_pairing_dao.class.php';

class GcImportService
{
    private $status;

    public function __construct() {}

    /**
     * Récupère le préfixe des options en fonction de la clé publique du site.
     *
     * @return string
     */
    public static function getOptionsPrefix() {
        $public_key = GcParamsService::getInstance()->graphcommentGetWebsiteId();
        return 'gc_import_' . $public_key . '_';
    }

    private static $allowed_status = array('pending', 'error', 'finished');

    /**
     * Migre les anciennes importations.
     */
    public function migrateOldImport() {
        $this->setImportStatus('finished');

        $date_begin = get_option('gc_import_date_begin');
        $this->setDateBegin($date_begin);

        $date_end = get_option('gc_import_date_end');
        $this->setDateEnd($date_end);

        $total = get_option('gc_import_total');
        $this->setTotalComments($total);

        $nbr_comments = get_option('gc_import_nbr_comment_import');
        $this->setNbrCommentsImport($nbr_comments);
    }

    /**
     * Définit le statut de l'importation.
     *
     * @param string $status
     * @return bool
     */
    private function setImportStatus($status) {
        if (in_array($status, self::$allowed_status, true)) {
            update_option(self::getOptionsPrefix() . 'status', $status);
            return true;
        }
        return false;
    }

    /**
     * Définit le message d'erreur de l'importation.
     *
     * @param string $msg
     */
    public function setImportErrorMsg($msg) {
        update_option(self::getOptionsPrefix() . 'error_msg', $msg);
    }

    /**
     * Récupère le message d'erreur de l'importation.
     *
     * @return string
     */
    public function getImportErrorMsg() {
        return get_option(self::getOptionsPrefix() . 'error_msg', '');
    }

    /**
     * Supprime le message d'erreur de l'importation.
     */
    private function deleteImportErrorMsg() {
        delete_option(self::getOptionsPrefix() . 'error_msg');
    }

    /**
     * Définit le numéro de lot de l'importation.
     *
     * @param int $batch_number
     */
    private function setImportBatchNumber($batch_number) {
        update_option(self::getOptionsPrefix() . 'batch_number', $batch_number);
    }

    /**
     * Récupère le statut actuel de l'importation.
     *
     * @return string
     */
    public function getImportStatus() {
        return get_option(self::getOptionsPrefix() . 'status', '');
    }

    /**
     * Récupère le numéro de lot de l'importation.
     *
     * @return int
     */
    private function getImportBatchNumber() {
        return get_option(self::getOptionsPrefix() . 'batch_number', 0);
    }

    /**
     * Définit le nombre total de commentaires à importer.
     *
     * @param int $nbr
     */
    private function setTotalComments($nbr) {
        update_option(self::getOptionsPrefix() . 'total', intval($nbr));
    }

    /**
     * Récupère le nombre total de commentaires à importer.
     *
     * @return int
     */
    public function getTotalComments() {
        return get_option(self::getOptionsPrefix() . 'total', 0);
    }

    /**
     * Récupère l'avancement de l'importation pour l'AJAX.
     *
     * @return array
     */
    public function getAjaxAdvancement() {
        $nbr_comment_import = intval(get_option(self::getOptionsPrefix() . 'nbr_comment_import', 0));
        $total_comments = intval($this->getTotalComments());
        $percent = $total_comments > 0 ? ($nbr_comment_import / $total_comments) * 100 : 0;
        return array(
            'nbr_comment_import' => $nbr_comment_import,
            'percent' => $percent,
            'status' => $this->getImportStatus(),
        );
    }

    /**
     * Récupère le nombre de commentaires importés.
     *
     * @return int
     */
    public function getNbrCommentsImport() {
        return intval(get_option(self::getOptionsPrefix() . 'nbr_comment_import', 0));
    }

    /**
     * Définit le nombre de commentaires importés.
     *
     * @param int $nbr
     */
    private function setNbrCommentsImport($nbr) {
        update_option(self::getOptionsPrefix() . 'nbr_comment_import', intval($nbr));
    }

    /**
     * Ajoute un nombre de commentaires importés au total actuel.
     *
     * @param int $nbr
     */
    private function addNbrCommentsImport($nbr) {
        $current = $this->getNbrCommentsImport();
        $this->setNbrCommentsImport($current + intval($nbr));
    }

    /**
     * Définit la date de début de l'importation.
     *
     * @param string|null $date
     */
    private function setDateBegin($date = null) {
        $date = $date ?: current_time('mysql');
        update_option(self::getOptionsPrefix() . 'date_begin', $date);
    }

    /**
     * Récupère la date de début de l'importation.
     *
     * @return string
     */
    public function getDateBegin() {
        return get_option(self::getOptionsPrefix() . 'date_begin', '');
    }

    /**
     * Définit la date de fin de l'importation.
     *
     * @param string|null $date
     */
    private function setDateEnd($date = null) {
        $date = $date ?: current_time('mysql');
        update_option(self::getOptionsPrefix() . 'date_end', $date);
    }

    /**
     * Récupère la date de fin de l'importation.
     *
     * @return string
     */
    public function getDateEnd() {
        return get_option(self::getOptionsPrefix() . 'date_end', '');
    }

    /**
     * Définit une erreur d'importation.
     *
     * @param string $msg
     */
    public function setImportationError($msg) {
        $this->setImportStatus('error');
        $this->setImportErrorMsg($msg);
    }

    /**
     * Convertit le statut du commentaire WordPress en statut GraphComment.
     *
     * @param string $comment_status
     * @return string
     */
    private static function getGcStatus($comment_status) {
        switch ($comment_status) {
            case '0':
                return 'pending';
            case '1':
                return 'approved';
            default:
                return 'deleted';
        }
    }

    /**
     * Initialise l'importation des commentaires.
     *
     * @return string|bool
     */
    public function importInit() {
        $this->status = $this->getImportStatus();

        if ($this->status !== '') {
            GcLogger::getLogger()->error('GcImportService::importInit() - An import is already pending');
            return __('An import is already in progress.', 'graphcomment-comment-system');
        }

        $nbr_comments = $this->getNbrCommentToImport();
        if ($nbr_comments === 0) {
            GcLogger::getLogger()->error('GcImportService::importInit() - No comment to import');
            return __('No comments to import.', 'graphcomment-comment-system');
        }

        $res = GcApiService::importInit($nbr_comments);

        if ($res['skip']) {
            return 'skip';
        }

        if ($res['error'] !== false) {
            return $res['error']; // importInit retourne le message d'erreur approprié
        }

        $batch_number = $res['batch_number'];

        delete_option(self::getOptionsPrefix() . 'status_stopped');
        $this->setImportBatchNumber($batch_number);
        $this->setImportStatus('pending');
        $this->setTotalComments($nbr_comments);
        $this->setNbrCommentsImport(0);
        $this->setDateBegin();

        return true;
    }

    /**
     * Redémarre l'importation.
     *
     * @return string|bool
     */
    public function restartImportationInit() {
        $public_key = get_option('gc_public_key');
        $batch_number = $this->getImportBatchNumber();

        if (!empty($batch_number)) {
            $response = wp_remote_post(
                API_URL_IMPORT_RESTART,
                array(
                    'sslverify' => SSLVERIFY,
                    'body' => array(
                        'public_key' => $public_key,
                        'platform' => 'wp',
                        'batch_import_number' => $batch_number,
                    ),
                )
            );

            $http_code = wp_remote_retrieve_response_code($response);

            if ($http_code !== 200) {
                GcLogger::getLogger()->error('GcImportService::restartImportationInit() - HTTP response code not 200 (URL: ' . API_URL_IMPORT_RESTART . ')');
                return __('Error restarting the import.', 'graphcomment-comment-system');
            }

            $this->deleteImportErrorMsg();
            $this->setNbrCommentsImport(0);
            $this->setDateBegin();
            $this->setImportStatus('pending');

            return true;
        }

        return __('No import found to restart.', 'graphcomment-comment-system');
    }

    /**
     * Arrête l'importation en cours.
     *
     * @return string|bool
     */
    public function stopImportation() {
        $public_key = get_option('gc_public_key');
        $res = GcApiService::importCancel($public_key);

        if ($res['error'] !== false) {
            return $res['error']; // importCancel retourne le message d'erreur approprié
        }

        $this->cleanImport();
        return true;
    }

    /**
     * Nettoie les données de l'importation.
     */
    private function cleanImport() {
        $this->deleteImportErrorMsg();
        update_option(self::getOptionsPrefix() . 'status_stopped', 'true');
        delete_option(self::getOptionsPrefix() . 'status');
        delete_option(self::getOptionsPrefix() . 'batch_number');
        delete_option(self::getOptionsPrefix() . 'nbr_comment_import');
        delete_option(self::getOptionsPrefix() . 'total');
    }

    /**
     * Récupère le nombre de commentaires à importer.
     *
     * @return int
     */
    private function getNbrCommentToImport() {
        global $wpdb;

        $excluded_post_types = array('product'); // Exclure les commentaires des produits WooCommerce

        $placeholders = implode(',', array_fill(0, count($excluded_post_types), '%s'));

        $sql = $wpdb->prepare(
            "SELECT COUNT(DISTINCT c.comment_ID) AS nbr
            FROM {$wpdb->comments} AS c
            LEFT JOIN " . GcCommentPairingDao::getTableName() . " AS gcp
                ON gcp.wp_comment_id = c.comment_ID
            LEFT JOIN {$wpdb->posts} AS p
                ON p.ID = c.comment_post_ID
            WHERE gcp.wp_comment_id IS NULL
                AND p.post_type NOT IN ($placeholders)",
            $excluded_post_types
        );

        $nbr = $wpdb->get_var($sql);

        if ($nbr === null) {
            GcLogger::getLogger()->error('GcImportService::getNbrCommentToImport() - Error executing SQL query.');
            return 0;
        }

        return intval($nbr);
    }

    /**
     * Termine l'importation en cours.
     *
     * @return bool
     */
    private function finishImport() {
        $public_key = get_option('gc_public_key');
        $batch_number = $this->getImportBatchNumber();

        if (empty($public_key) || empty($batch_number)) {
            GcLogger::getLogger()->error('finishImport - Missing gc_public_key or batch_number');
            return false;
        }

        $this->setDateEnd();
        $this->setImportStatus('finished');

        $res = GcApiService::importFinish($public_key, $batch_number);

        if ($res['error'] !== false) {
            return false;
        }

        $this->deleteImportErrorMsg();

        return true;
    }

    /**
     * Importe le prochain lot de commentaires.
     *
     * @return bool
     */
    public function importNextComments() {
        global $wpdb;

        $fname = 'GcImportService::importNextComments()';

        $offset = $this->getNbrCommentsImport();
        $limit = NUMBER_COMMENTS_IMPORT_PARTS;

        $excluded_post_types = array('product'); // Exclure les commentaires des produits WooCommerce

        $placeholders = implode(',', array_fill(0, count($excluded_post_types), '%s'));

        $sql = $wpdb->prepare(
            "SELECT c.*
            FROM {$wpdb->comments} AS c
            LEFT JOIN " . GcCommentPairingDao::getTableName() . " AS gcp
                ON gcp.wp_comment_id = c.comment_ID
            LEFT JOIN {$wpdb->posts} AS p
                ON p.ID = c.comment_post_ID
            WHERE gcp.wp_comment_id IS NULL
                AND p.post_type NOT IN ($placeholders)
            ORDER BY c.comment_ID ASC
            LIMIT %d, %d",
            array_merge($excluded_post_types, array($offset, $limit))
        );

        GcLogger::getLogger()->debug($fname . ' - SQL Query: ' . $sql);

        $comments = $wpdb->get_results($sql);

        if ($comments === null) {
            GcLogger::getLogger()->error($fname . ' - Error executing SQL query: ' . $wpdb->last_error);
            $this->setImportationError(__('Error importing comments.', 'graphcomment-comment-system'));
            return false;
        }

        if (empty($comments)) {
            GcLogger::getLogger()->debug($fname . ' - No more comments to import.');
            // Importation terminée
            $this->finishImport();
            return false;
        }

        // Récupération des posts associés aux commentaires
        $post_ids = wp_list_pluck($comments, 'comment_post_ID');
        $posts = get_posts(array(
            'include' => $post_ids,
            'post_type' => 'any',
            'post_status' => 'any',
        ));

        $posts_by_id = array();
        foreach ($posts as $post) {
            $posts_by_id[$post->ID] = $post;
        }

        // Préparation des commentaires pour l'envoi
        $formatted_comments = array();

        foreach ($comments as $comment) {
            $post = isset($posts_by_id[$comment->comment_post_ID]) ? $posts_by_id[$comment->comment_post_ID] : null;

            if (!$post) {
                continue; // Ignorer si le post n'est pas trouvé
            }

            $formatted_comments[] = array(
                'id' => $comment->comment_ID,
                'parent_id' => $comment->comment_parent ? $comment->comment_parent : null,
                'content' => $comment->comment_content,
                'status' => self::getGcStatus($comment->comment_approved),
                'spam' => ($comment->comment_approved === 'spam' ? '1' : null),
                'author_username' => $comment->comment_author ?: null,
                'author_email' => $comment->comment_author_email ?: null,
                'author_ip' => $comment->comment_author_IP ?: null,
                'date' => $comment->comment_date_gmt,
                'page' => get_permalink($post),
                'page_date' => $post->post_date,
                'identifier' => GcParamsService::getInstance()->graphcommentIdentifierGenerate($post->ID),
                'page_title' => get_the_title($post),
                'uid' => $post->ID,
                'guid' => get_the_guid($post),
            );
        }

        GcLogger::getLogger()->debug($fname . ' - Number of comments to send: ' . count($formatted_comments));

        if (empty($formatted_comments)) {
            GcLogger::getLogger()->debug($fname . ' - No valid comments found to import.');
            // Importation terminée
            $this->finishImport();
            return false;
        }

        $res = GcApiService::pushImportComments($this->getImportBatchNumber(), $formatted_comments);

        if ($res['error'] !== false) {
            if (is_string($res['error'])) {
                $this->setImportationError($res['error']);
            }
            return false;
        }

        // Mettre à jour le nombre de commentaires importés
        $this->addNbrCommentsImport(count($formatted_comments));

        return true;
    }
}
