<?php

class GcApiService {

    /**
     * Signale les erreurs HTTP au journal.
     *
     * @param string $url       L'URL qui a été appelée.
     * @param int    $http_code Le code de réponse HTTP.
     * @param string $message   Message d'erreur supplémentaire.
     */
    private static function report_http_error( $url, $http_code = null, $message = '' ) {
        $dbt    = debug_backtrace( DEBUG_BACKTRACE_IGNORE_ARGS, 2 );
        $caller = isset( $dbt[1]['function'] ) ? $dbt[1]['function'] : 'Fonction inconnue';
        $log    = sprintf(
            '%s - Erreur HTTP%s (URL: %s)',
            $caller,
            $http_code ? ' ' . $http_code : '',
            esc_url_raw( $url )
        );
        if ( $message ) {
            $log .= ' - ' . sanitize_text_field( $message );
        }
        GcLogger::getLogger()->error( $log );
    }

    /**
     * Effectue une requête API vers l'URL spécifiée avec les arguments donnés.
     *
     * @param string $method   La méthode HTTP ('GET' ou 'POST').
     * @param string $url      L'URL de l'API.
     * @param array  $args     Les arguments de la requête.
     * @return array           Un tableau contenant 'response' ou 'error'.
     */
    private static function make_api_request( $method, $url, $args = array() ) {
        $default_args = array(
            'sslverify' => defined( 'SSLVERIFY' ) ? SSLVERIFY : true,
            'timeout'   => 50,
        );

        $args = wp_parse_args( $args, $default_args );

        if ( 'POST' === strtoupper( $method ) ) {
            $response = wp_remote_post( $url, $args );
        } else {
            $response = wp_remote_get( $url, $args );
        }

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            self::report_http_error( $url, null, $error_message );
            return array( 'error' => $error_message );
        }

        $http_code = wp_remote_retrieve_response_code( $response );
        $body      = wp_remote_retrieve_body( $response );

        if ( $http_code !== 200 ) {
            self::report_http_error( $url, $http_code, $body );
            return array( 'error' => sprintf( __( 'La requête API a échoué avec le code %d.', 'graphcomment-comment-system' ), $http_code ) );
        }

        return array( 'response' => $body );
    }

    /**
     * Initialise une importation avec l'API GC.
     *
     * @param int $comment_count Le nombre de commentaires à importer.
     * @return array Résultat de l'appel API.
     */
    public static function importInit( $comment_count ) {
        $api_url = defined( 'API_URL_IMPORT_INIT' ) ? API_URL_IMPORT_INIT : '';
        if ( empty( $api_url ) ) {
            return array( 'error' => __( 'URL de l\'API non définie.', 'graphcomment-comment-system' ), 'batch_number' => null );
        }

        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . GcParamsService::getInstance()->graphcommentGetClientToken(),
            ),
            'body'    => array(
                'public_key' => GcParamsService::getInstance()->graphcommentGetWebsite(),
                'platform'   => 'wp',
                'total'      => intval( $comment_count ),
            ),
        );

        $result = self::make_api_request( 'POST', $api_url, $args );

        if ( isset( $result['error'] ) ) {
            return array( 'error' => $result['error'], 'batch_number' => null );
        }

        $body = json_decode( $result['response'] );

        if ( ! isset( $body->batch_import_number ) ) {
            self::report_http_error( $api_url, null, 'batch_import_number manquant dans la réponse.' );
            return array( 'error' => __( 'Réponse API invalide.', 'graphcomment-comment-system' ), 'batch_number' => null );
        }

        GcLogger::getLogger()->debug( 'GcApiService::import_init() - Importation initialisée avec succès.' );

        return array( 'error' => false, 'batch_number' => sanitize_text_field( $body->batch_import_number ) );
    }

    /**
     * Annule une importation en cours via l'API GC.
     *
     * @param string $public_key La clé publique du site.
     * @return array Résultat de l'appel API.
     */
    public static function importCancel( $public_key ) {
        $api_url = defined( 'API_URL_IMPORT_STOP' ) ? API_URL_IMPORT_STOP : '';
        if ( empty( $api_url ) ) {
            return array( 'error' => __( 'URL de l\'API non définie.', 'graphcomment-comment-system' ) );
        }

        $args = array(
            'body' => array(
                'public_key' => sanitize_text_field( $public_key ),
                'platform'   => 'wp',
            ),
        );

        $result = self::make_api_request( 'POST', $api_url, $args );

        if ( isset( $result['error'] ) ) {
            return array( 'error' => __( 'Erreur lors de l\'annulation de l\'importation.', 'graphcomment-comment-system' ) );
        }

        return array( 'error' => false );
    }

    /**
     * Envoie plus de commentaires à importer via l'API GC.
     *
     * @param string $batch_number Le numéro de lot d'importation.
     * @param array  $comments     Les commentaires à importer.
     * @return array Résultat de l'appel API.
     */
    public static function pushImportComments( $batch_number, $comments ) {
        $api_url = defined( 'API_URL_IMPORT' ) ? API_URL_IMPORT : '';
        if ( empty( $api_url ) ) {
            return array( 'error' => __( 'URL de l\'API non définie.', 'graphcomment-comment-system' ) );
        }

        $args = array(
            'body' => array(
                'public_key'         => GcParamsService::getInstance()->graphcommentGetWebsite(),
                'platform'           => 'wp',
                'batch_import_number' => sanitize_text_field( $batch_number ),
                'comments'           => $comments,
            ),
        );

        $result = self::make_api_request( 'POST', $api_url, $args );

        if ( isset( $result['error'] ) ) {
            if ( get_option( GcImportService::getOptionsPrefix() . 'status_stopped' ) === 'true' ) {
                return array( 'error' => true );
            }
            return array( 'error' => __( 'Erreur lors de l\'importation des commentaires.', 'graphcomment-comment-system' ) );
        }

        $body = $result['response'];

        if ( 'import canceled' === $body ) {
            self::report_http_error( $api_url, null, 'Importation annulée.' );
            return array( 'error' => true );
        }

        return array( 'error' => false );
    }

    /**
     * Termine l'importation via l'API GC.
     *
     * @param string $public_key   La clé publique du site.
     * @param string $batch_number Le numéro de lot d'importation.
     * @return array Résultat de l'appel API.
     */
    public static function importFinish( $public_key, $batch_number ) {
        $api_url = defined( 'API_URL_IMPORT_FINISH' ) ? API_URL_IMPORT_FINISH : '';
        if ( empty( $api_url ) ) {
            return array( 'error' => true );
        }

        $args = array(
            'body' => array(
                'public_key'         => sanitize_text_field( $public_key ),
                'platform'           => 'wp',
                'batch_import_number' => sanitize_text_field( $batch_number ),
            ),
        );

        $result = self::make_api_request( 'POST', $api_url, $args );

        if ( isset( $result['error'] ) ) {
            return array( 'error' => true );
        }

        return array( 'error' => false );
    }

    /**
     * Récupère tous les nouveaux commentaires via l'API GC.
     *
     * @return array Résultat de l'appel API.
     */
    public static function getNewComments( ) {
        $api_url = defined( 'API_URL_SYNC_COMMENTS' ) ? API_URL_SYNC_COMMENTS : '';
        if ( empty( $api_url ) ) {
            return array( 'error' => true, 'comments' => null );
        }

        $pubkey = get_option( 'gc_api_public_key' );
        $priv_key = get_option( 'gc_api_private_key' );

        if ( empty( $pubkey ) || empty( $priv_key ) ) {
            self::report_http_error( $api_url, null, 'Clés API manquantes.' );
            update_option( 'gc-sync-error', json_encode( array( 'content' => __( 'Clés API manquantes.', 'graphcomment-comment-system' ) ) ) );
            return array( 'error' => true, 'comments' => null );
        }

        $key = generateSsoData( 'key', $priv_key );

        $url = str_replace( ':pubkey', $pubkey, str_replace( ':key', $key, $api_url ) );

        $result = self::make_api_request( 'GET', $url );

        if ( isset( $result['error'] ) ) {
            update_option( 'gc-sync-error', json_encode( array( 'content' => __( 'Erreur lors de la récupération de la synchronisation.', 'graphcomment-comment-system' ) ) ) );
            GcLogger::getLogger()->error( 'Gc sync: erreur (get_new_comments) ' . $result['error'] );
            return array( 'error' => true, 'comments' => null );
        }

        $body = json_decode( $result['response'] );

        if ( ! isset( $body->comments ) ) {
            self::report_http_error( $url, null, 'Aucun commentaire trouvé dans la réponse.' );
            return array( 'error' => true, 'comments' => null );
        }

        return array( 'error' => false, 'comments' => $body->comments );
    }

    /**
     * Confirme que les nouveaux commentaires ont été mis à jour pour ne plus les recevoir.
     *
     * @param array  $comment_ids   Les identifiants des commentaires à confirmer.
     * @return array Résultat de l'appel API.
     */
    public static function confirmNewComments( $comment_ids ) {
        $api_url = defined( 'API_URL_SYNC_COMMENTS' ) ? API_URL_SYNC_COMMENTS : '';
        if ( empty( $api_url ) ) {
            return array( 'error' => true );
        }

        $pubkey = get_option( 'gc_api_public_key' );
        $priv_key = get_option( 'gc_api_private_key' );

        if ( empty( $pubkey ) || empty( $priv_key ) ) {
            self::report_http_error( $api_url, null, 'Clés API manquantes.' );
            update_option( 'gc-sync-error', json_encode( array( 'content' => __( 'Clés API manquantes.', 'graphcomment-comment-system' ) ) ) );
            return array( 'error' => true );
        }

        $key = generateSsoData( $comment_ids, $priv_key );

        $url = str_replace( ':pubkey', $pubkey, str_replace( ':key', $key, $api_url ) );

        $result = self::make_api_request( 'POST', $url );

        if ( isset( $result['error'] ) ) {
            update_option( 'gc-sync-error', json_encode( array( 'content' => __( 'Erreur lors de la confirmation des commentaires.', 'graphcomment-comment-system' ) ) ) );
            GcLogger::getLogger()->error( 'Gc sync: erreur (confirm_new_comments) ' . $result['error'] );
            return array( 'error' => true );
        }

        return array( 'error' => false );
    }
}
