<?php

class GcCommentPairingDao
{
    private static $table_name = 'graphcomment_comment_pairing';
    private $known_ids = array();

    public static function getTableName()
    {
        global $wpdb;
        return $wpdb->prefix . self::$table_name;
    }

    public static function createPairingTable()
    {
        global $wpdb;

        $table_name = self::getTableName();
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id MEDIUMINT(9) NOT NULL AUTO_INCREMENT,
            wp_comment_id MEDIUMINT(9) NOT NULL,
            gc_comment_id VARCHAR(255) NOT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY wp_gc_pair (wp_comment_id, gc_comment_id),
            INDEX gc_comment_id_idx (gc_comment_id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        if ($wpdb->last_error !== '') {
            $log = 'GcCommentPairingDao::createPairingTable() - Error SQL: (' . $sql . ') - last_error: (' . $wpdb->last_error . ')';
            GcLogger::getLogger()->error($log);
            return false;
        }
        return true;
    }

    private function addKnownIds($new_known_ids)
    {
        if (is_array($new_known_ids) && !empty($new_known_ids)) {
            $this->known_ids = array_merge($this->known_ids, $new_known_ids);
        }
    }

    public function findIds($ids)
    {
        global $wpdb;

        if (empty($ids)) {
            return array();
        }

        $placeholders = array_fill(0, count($ids), '%s');
        $values = array_map('sanitize_text_field', $ids);

        $placeholders_string = implode(', ', $placeholders);
        $sql = "SELECT `wp_comment_id`, `gc_comment_id` FROM " . self::getTableName() . " WHERE `gc_comment_id` IN ($placeholders_string)";

        $prepared_query = $wpdb->prepare($sql, $values);
        $new_known_ids = $wpdb->get_results($prepared_query, ARRAY_A);

        if ($new_known_ids === false) {
            GcLogger::getLogger()->error('GcCommentPairingDao::findIds() - Error SQL: (' . $prepared_query . ') - last_error: (' . $wpdb->last_error . ')');
            return array();
        }

        $this->addKnownIds($new_known_ids);

        return $new_known_ids;
    }

    public function findWordpressId($graphCommentId)
    {
        // Chercher d'abord dans les IDs connus
        if (!empty($this->known_ids)) {
            foreach ($this->known_ids as $pair) {
                if ($pair['gc_comment_id'] === $graphCommentId) {
                    return $pair['wp_comment_id'];
                }
            }
        }

        // Si l'ID n'est pas trouvé dans les IDs connus, chercher dans la base de données
        global $wpdb;
        $sql = "SELECT `wp_comment_id` FROM " . self::getTableName() . " WHERE `gc_comment_id` = %s";
        $prepared_query = $wpdb->prepare($sql, sanitize_text_field($graphCommentId));
        $wp_comment_id = $wpdb->get_var($prepared_query);

        if ($wp_comment_id === null) {
            // Pas trouvé dans la base de données non plus
            return false;
        }

        // Si trouvé, ajouter à la liste des IDs connus pour de futures recherches
        $this->addKnownIds(array(array('wp_comment_id' => intval($wp_comment_id), 'gc_comment_id' => sanitize_text_field($graphCommentId))));

        return $wp_comment_id;
    }

    public function insertKnownPairIds($wordpress_id, $graphcomment_id)
    {
        global $wpdb;

        $table_name = self::getTableName();
        $sql = "INSERT INTO $table_name (wp_comment_id, gc_comment_id) VALUES (%d, %s)";
        $prepared_query = $wpdb->prepare($sql, intval($wordpress_id), sanitize_text_field($graphcomment_id));
        $ret = $wpdb->query($prepared_query);

        if ($ret === false) {
            GcLogger::getLogger()->error('GcCommentPairingDao::insertKnownPairIds() - Error SQL: (' . $prepared_query . ') - last_error: (' . $wpdb->last_error . ')');
            return false;
        }

        // Enregistrer le couple WP_ID <=> GC_ID dans le tableau local $known_ids
        $this->addKnownIds(array(array('wp_comment_id' => intval($wordpress_id), 'gc_comment_id' => sanitize_text_field($graphcomment_id))));
        return true;
    }
}
