<?php

if (!class_exists('GcIframeManager')) {
    class GcIframeManager
    {
        const GRAPHCOMMENT_ADMIN_IFRAME = 0;
        const GRAPHCOMMENT_CONNEXION_IFRAME = 1;
        const GRAPHCOMMENT_DISCONNEXION_IFRAME = 2;

        /**
         * Retrieve the iframe URL based on the iframe type.
         *
         * @param int $iframe_name Type of iframe.
         * @return string The URL of the iframe.
         */
        public static function getIframeUrl($iframe_name)
        {
            $gcParams = GcParamsService::getInstance();
            $pluginData = get_plugin_data(dirname(__FILE__) . '/../../graphcomment.php');

            // Ensure the necessary constants are defined.
            if (!defined('API_URL') || !defined('ADMIN_URL')) {
                GcLogger::getLogger()->error('API_URL or ADMIN_URL is not defined.');
                return '';
            }

            switch ($iframe_name) {
                case self::GRAPHCOMMENT_CONNEXION_IFRAME:
                    return self::getConnexionIframeUrl($gcParams);

                case self::GRAPHCOMMENT_DISCONNEXION_IFRAME:
                    return self::getDeconnexionIframeUrl();

                case self::GRAPHCOMMENT_ADMIN_IFRAME:
                default:
                    return self::getAdminIframeUrl($gcParams, $pluginData);
            }
        }

        /**
         * Generate the URL for the Connexion iframe.
         *
         * @param object $gcParams GcParamsService instance.
         * @return string Connexion iframe URL.
         */
        private static function getConnexionIframeUrl($gcParams)
        {
            $clientKey = $gcParams->graphcommentGetClientKey();
            $redirectUri = urlencode($gcParams->graphcommentGetRedirectUri());

            return constant('API_URL') . '/oauth/authorize?response_type=code&client_id=' . $clientKey . '&redirect_uri=' . $redirectUri . '&scope=*';
        }

        /**
         * Generate the URL for the Deconnexion iframe.
         *
         * @return string Deconnexion iframe URL.
         */
        private static function getDeconnexionIframeUrl()
        {
            return constant('ADMIN_URL') . '/#/logout';
        }

        /**
         * Generate the URL for the Admin iframe.
         *
         * @param object $gcParams GcParamsService instance.
         * @param array $pluginData Plugin data array.
         * @return string Admin iframe URL.
         */
        private static function getAdminIframeUrl($gcParams, $pluginData)
        {
            // Sanitize the URL parameter from GET request.
            $url = isset($_GET['url']) ? sanitize_text_field($_GET['url']) : '';

            // Construct the admin iframe URL.
            $baseUrl = constant('ADMIN_URL') . '/#/website/';
            $websiteId = ($gcParams->graphcommentHasWebsites() && $gcParams->graphcommentIsWebsiteChoosen())
                ? $gcParams->graphcommentGetWebsiteId() . '/graphcomment'
                : 'new';

            return $baseUrl . $websiteId . $url . '?iframe=wp&pluginVersion=' . $pluginData['Version'];
        }
    }
}
