<?php

/**
 * Class GcMenu
 *
 * This class manages the addition of the GraphComment menu in the WordPress admin dashboard.
 */

if (!class_exists('GcMenu')) {
    class GcMenu
    {
        /**
         * Constructor that binds the create_menu method to a specified WordPress action hook.
         *
         * @param string $action_name The WordPress action hook name.
         */
        public function __construct($action_name)
        {
            // Bind the create_menu method to the specified action.
            add_action($action_name, [$this, 'create_menu']);
        }

        /**
         * Generate the notification badge HTML.
         *
         * @param int $notifCount The number of notifications.
         * @param string $notifClass Optional CSS class for the notification.
         * @return string HTML string for the notification badge.
         */
        private function get_notif_str($notifCount, $notifClass = '')
        {
            return sprintf(
                '<span class="update-plugins count-%d" %s>
                    <span class="gc-notif-count %s">%d</span>
                </span>',
                $notifCount,
                $notifCount ? '' : 'style="display: none"',
                $notifClass,
                $notifCount
            );
        }

        /**
         * Generate the menu HTML with or without a notification badge.
         *
         * @param int $notifCount The number of notifications.
         * @param string $label The label for the menu item.
         * @return string HTML string for the menu label.
         */
        private function get_menu_html($notifCount, $label)
        {
            return $notifCount > 0
                ? '<span class="gc-menu-label gc-menu-label-truncate">' . $label . '</span>'
                : '<span class="gc-menu-label">' . $label . '</span>';
        }

        /**
         * Adds the GraphComment button to the WordPress admin menu.
         */
        public function create_menu()
        {
            // Fetch the current notifications.
            $this->get_notif();

            $GraphCommentNotif = $GraphCommentNotifSettings = 0;
            $nbrNewComments = get_option('gc_notif_comments');
            $gc_sync_error = get_option('gc-sync-error');

            if ($nbrNewComments !== false) {
                $GraphCommentNotif = intval($nbrNewComments);
            }

            // If the user is not logged in, show the GraphComment login menu with a notification.
            if (!GcParamsService::getInstance()->graphcommentOAuthIsLogged()) {
                $this->createLoginMenu($GraphCommentNotif);
                return;
            }

            // If not configured, add a notification to the settings.
            if (!GcParamsService::getInstance()->graphcommentHasWebsites() || !GcParamsService::getInstance()->graphcommentIsWebsiteChoosen()) {
                $GraphCommentNotifSettings = 1;
            }

            // Add the main GraphComment admin menu and submenus.
            $this->createAdminMenus($GraphCommentNotif, $GraphCommentNotifSettings);

            // Handle potential disconnection requests.
            if (isset($_GET['graphcomment-disconnect']) && $_GET['graphcomment-disconnect'] === 'true') {
                GcLogger::getLogger()->debug('GcMenu::create_menu() - User wants to disconnect');
                $this->createSubMenuSettings();
                return GcParamsService::getInstance()->graphcommentOAuthReLog(true, true);
            }

            // Handle scenarios where the user needs to create or choose a website.
            $this->handleWebsiteSelection();

            return null;
        }

        /**
         * Create the login menu with a notification when the user is not logged in.
         *
         * @param int $GraphCommentNotif The notification count for the main menu.
         */
        private function createLoginMenu($GraphCommentNotif)
        {
            add_menu_page(
                __('GC Plugin Admin Page Title', 'graphcomment-comment-system'),
                $this->get_menu_html(1, __('GC Plugin Admin Menu Title', 'graphcomment-comment-system')) . ' ' .
                $this->get_notif_str(1, 'gc-notif-main'),
                'manage_options',
                'graphcomment',
                '_graphcomment_settings_page_admin',
                plugins_url('../../theme/images/icon.png', __FILE__)
            );

            $page_url = admin_url('admin.php');
            $page_query = 'page=graphcomment&debug=redirectOAuth';

            if (isset($_GET['page']) && $_GET['page'] === 'graphcomment-settings' && !isset($_POST['gc_action'])) {
                add_submenu_page(
                    'graphcomment',
                    __('GC Plugin Settings Page Title', 'graphcomment-comment-system'),
                    __('GC Plugin Settings SubMenu Title', 'graphcomment-comment-system') . ' ' .
                    $this->get_notif_str(1, 'gc-notif-settings'),
                    'manage_options',
                    'graphcomment-settings',
                    '_graphcomment_settings_page'
                );

                wp_redirect($page_url . '?' . $page_query);
                exit;
            }
        }

        /**
         * Create the main admin menus for GraphComment.
         *
         * @param int $GraphCommentNotif The notification count for the main menu.
         * @param int $GraphCommentNotifSettings The notification count for the settings menu.
         */
        private function createAdminMenus($GraphCommentNotif, $GraphCommentNotifSettings)
        {
            add_menu_page(
                __('GC Plugin Admin Page Title', 'graphcomment-comment-system'),
                $this->get_menu_html($GraphCommentNotif + $GraphCommentNotifSettings, __('GC Plugin Admin Menu Title', 'graphcomment-comment-system')) . ' ' .
                $this->get_notif_str($GraphCommentNotif + $GraphCommentNotifSettings, 'gc-notif-main'),
                'manage_options',
                'graphcomment',
                '_graphcomment_settings_page_admin',
                plugins_url('../../theme/images/icon.png', __FILE__)
            );
            add_submenu_page(
                'graphcomment',
                __('GC Plugin Admin Page Title', 'graphcomment-comment-system'),
                __('GC Plugin Admin SubMenu Title', 'graphcomment-comment-system') . ' ' .
                $this->get_notif_str($GraphCommentNotif, 'gc-notif-admin'),
                'manage_options',
                'graphcomment'
            );
            add_submenu_page(
                'graphcomment',
                __('GC Plugin Settings Page Title', 'graphcomment-comment-system'),
                __('GC Plugin Settings SubMenu Title', 'graphcomment-comment-system') . ' ' .
                $this->get_notif_str($GraphCommentNotifSettings, 'gc-notif-settings'),
                'manage_options',
                'graphcomment-settings',
                '_graphcomment_settings_page'
            );
        }

        /**
         * Handle scenarios where the user needs to create or select a website.
         */
        private function handleWebsiteSelection()
        {
            if (!GcParamsService::getInstance()->graphcommentHasWebsites()) {
                GcLogger::getLogger()->debug('GcMenu::create_menu() - User has no website, needs to create one');
                if (GcParamsService::getInstance()->graphcommentHasUser()) {
                    $page_url = admin_url('admin.php');
                    $page_query = 'page=graphcomment&debug=hasNoWebsite';

                    if (isset($_GET['page']) && $_GET['page'] === 'graphcomment-settings' && !isset($_POST['gc_action'])) {
                        wp_redirect($page_url . '?' . $page_query);
                        exit;
                    }
                }
            } elseif (!GcParamsService::getInstance()->graphcommentIsWebsiteChoosen()) {
                if (isset($_GET['page']) && $_GET['page'] === 'graphcomment') {
                    if (get_option('gc_create_website') !== 'true') {
                        wp_redirect(admin_url('admin.php?page=graphcomment-settings'));
                        exit;
                    } else {
                        delete_option('gc_create_website');
                    }
                }
            }
        }

        /**
         * Check notifications and update the menu badge if there are any.
         *
         * @return bool True if notifications were successfully retrieved, false otherwise.
         */
        public function get_notif()
        {
            delete_option('gc_notif_comments');
            $gc_website_id = get_option('gc_website_id');

            if (!$gc_website_id) return false;

            $body = ['platform' => 'wp'];
            $url = constant('API_URL') . '/website/' . $gc_website_id . '/notif';
            $request = wp_remote_post($url, ['timeout' => 5, 'sslverify' => constant('SSLVERIFY'), 'body' => $body]);

            $httpCode = wp_remote_retrieve_response_code($request);
            $body = wp_remote_retrieve_body($request);

            if ($httpCode !== 200) {
                GcLogger::getLogger()->error('GcMenu::get_notif() - Got HTTP ret !== 200 (url: ' . $url . ')');
                update_option('gc-notif-error', json_encode(['content' => __('Error Getting Notif', 'graphcomment-comment-system')]));
                return false;
            }

            $ret = json_decode($body);
            if (!$ret || !isset($ret->comments)) {
                GcLogger::getLogger()->error('GcMenu::get_notif() - Invalid response or missing comments (url: ' . $url . ')');
                update_option('gc-notif-error', json_encode(['content' => __('Error Getting Notif', 'graphcomment-comment-system')]));
                return false;
            }

            if ($ret->comments === 0) {
                delete_option('gc_notif_comments');
            } else {
                update_option('gc_notif_comments', $ret->comments);
            }

            return true;
        }

        /**
         * Creates a dummy settings submenu to handle specific redirect scenarios.
         */
        private function createSubMenuSettings()
        {
            add_submenu_page(
                'graphcomment',
                __('GC Plugin Settings Page Title', 'graphcomment-comment-system'),
                __('GC Plugin Settings SubMenu Title', 'graphcomment-comment-system'),
                'manage_options',
                'graphcomment-settings',
                '_graphcomment_settings_page'
            );
        }
    }
}
